import logging
import time
from typing import Dict, List, Optional
from datetime import datetime
from database import SessionLocal, WhatsAppMessage, Business

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class WhatsAppSender:
    """
    نظام إرسال رسائل الواتساب
    
    ملاحظة: هذا الكلاس يحتاج إلى ربط مع خدمة واتساب مثل:
    - Twilio API for WhatsApp
    - WhatsApp Business API
    - حل مخصص مثل Baileys أو whatsapp-web.js
    
    الكود الحالي يوفر البنية الأساسية ويمكن تطويره حسب الخدمة المستخدمة
    """
    
    def __init__(self):
        self.db = SessionLocal()
        self.delay_between_messages = 3  # ثواني بين كل رسالة
    
    def send_message(self, phone_number: str, message: str, business_id: Optional[int] = None) -> Dict:
        """
        إرسال رسالة واتساب
        
        Args:
            phone_number: رقم الهاتف بصيغة دولية
            message: نص الرسالة
            business_id: معرف العمل التجاري (اختياري)
            
        Returns:
            نتيجة الإرسال
        """
        try:
            # تنظيف رقم الهاتف
            cleaned_phone = self._clean_phone_number(phone_number)
            
            if not cleaned_phone:
                return {
                    'success': False,
                    'error': 'رقم هاتف غير صالح'
                }
            
            # هنا يتم الربط مع API الواتساب الفعلي
            # المثال الحالي يوفر محاكاة للإرسال
            
            logger.info(f"📱 إرسال رسالة إلى: {cleaned_phone}")
            
            # محاكاة الإرسال (استبدل هذا بالكود الفعلي)
            success = self._simulate_send(cleaned_phone, message)
            
            # حفظ السجل في قاعدة البيانات
            whatsapp_msg = WhatsAppMessage(
                business_id=business_id,
                phone_number=cleaned_phone,
                message_content=message,
                status='sent' if success else 'failed',
                sent_at=datetime.utcnow(),
                error_message=None if success else 'فشل الإرسال'
            )
            
            self.db.add(whatsapp_msg)
            self.db.commit()
            
            if success:
                logger.info(f"✅ تم إرسال الرسالة بنجاح إلى {cleaned_phone}")
                return {
                    'success': True,
                    'message_id': whatsapp_msg.id,
                    'phone': cleaned_phone
                }
            else:
                logger.error(f"❌ فشل إرسال الرسالة إلى {cleaned_phone}")
                return {
                    'success': False,
                    'error': 'فشل الإرسال'
                }
            
        except Exception as e:
            logger.error(f"❌ خطأ في إرسال الرسالة: {str(e)}")
            return {
                'success': False,
                'error': str(e)
            }
    
    def send_bulk_messages(self, messages: List[Dict]) -> Dict:
        """
        إرسال رسائل متعددة
        
        Args:
            messages: قائمة بالرسائل، كل عنصر يحتوي على phone و message و business_id
            
        Returns:
            إحصائيات الإرسال
        """
        results = {
            'total': len(messages),
            'sent': 0,
            'failed': 0,
            'details': []
        }
        
        for msg_data in messages:
            phone = msg_data.get('phone')
            message = msg_data.get('message')
            business_id = msg_data.get('business_id')
            
            if not phone or not message:
                results['failed'] += 1
                continue
            
            result = self.send_message(phone, message, business_id)
            
            if result.get('success'):
                results['sent'] += 1
            else:
                results['failed'] += 1
            
            results['details'].append({
                'phone': phone,
                'success': result.get('success'),
                'error': result.get('error')
            })
            
            # انتظار بين الرسائل لتجنب الحظر
            time.sleep(self.delay_between_messages)
        
        logger.info(f"📊 إحصائيات الإرسال: {results['sent']} نجح، {results['failed']} فشل")
        return results
    
    def _clean_phone_number(self, phone: str) -> Optional[str]:
        """تنظيف رقم الهاتف وتحويله للصيغة الدولية"""
        if not phone:
            return None
        
        # إزالة الأحرف غير الرقمية
        cleaned = ''.join(filter(str.isdigit, phone))
        
        # التحقق من وجود رقم صالح
        if len(cleaned) < 10:
            return None
        
        # إضافة رمز الدولة إذا لم يكن موجوداً
        if not cleaned.startswith('966'):  # السعودية كمثال
            if cleaned.startswith('0'):
                cleaned = '966' + cleaned[1:]
            else:
                cleaned = '966' + cleaned
        
        return '+' + cleaned
    
    def _simulate_send(self, phone: str, message: str) -> bool:
        """
        إرسال رسالة WhatsApp فعلياً
        """
        # محاولة استخدام Twilio إذا كانت المفاتيح موجودة
        try:
            import os
            from dotenv import load_dotenv
            load_dotenv()
            
            account_sid = os.getenv('TWILIO_ACCOUNT_SID')
            auth_token = os.getenv('TWILIO_AUTH_TOKEN')
            whatsapp_number = os.getenv('TWILIO_WHATSAPP_NUMBER', 'whatsapp:+14155238886')
            
            if account_sid and auth_token:
                # استخدام Twilio للإرسال الفعلي
                try:
                    from twilio.rest import Client
                    client = Client(account_sid, auth_token)
                    
                    msg = client.messages.create(
                        from_=whatsapp_number,
                        body=message,
                        to=f'whatsapp:{phone}'
                    )
                    
                    logger.info(f"✅ تم إرسال الرسالة عبر Twilio - SID: {msg.sid}")
                    return msg.sid is not None
                    
                except ImportError:
                    logger.warning("⚠️ مكتبة Twilio غير مثبتة، استخدم: pip install twilio")
                    return self._simulate_only(phone, message)
                except Exception as e:
                    logger.error(f"❌ خطأ في Twilio: {str(e)}")
                    return False
            else:
                # لا توجد مفاتيح Twilio - محاكاة فقط
                logger.info("⚠️ لا توجد مفاتيح Twilio - وضع المحاكاة")
                return self._simulate_only(phone, message)
                
        except Exception as e:
            logger.error(f"❌ خطأ في الإرسال: {str(e)}")
            return False
    
    def _simulate_only(self, phone: str, message: str) -> bool:
        """محاكاة الإرسال فقط (للاختبار)"""
        print(f"\n{'='*60}")
        print(f"📱 رسالة واتساب جديدة (محاكاة)")
        print(f"{'='*60}")
        print(f"إلى: {phone}")
        print(f"الرسالة:\n{message}")
        print(f"{'='*60}\n")
        print("💡 لإرسال فعلي: أضف مفاتيح Twilio في ملف .env")
        print(f"{'='*60}\n")
        return True
    
    def get_message_status(self, message_id: int) -> Optional[Dict]:
        """الحصول على حالة رسالة معينة"""
        msg = self.db.query(WhatsAppMessage).filter_by(id=message_id).first()
        return msg.to_dict() if msg else None
    
    def get_campaign_stats(self, business_ids: List[int]) -> Dict:
        """الحصول على إحصائيات الحملة"""
        total = self.db.query(WhatsAppMessage).filter(
            WhatsAppMessage.business_id.in_(business_ids)
        ).count()
        
        sent = self.db.query(WhatsAppMessage).filter(
            WhatsAppMessage.business_id.in_(business_ids),
            WhatsAppMessage.status == 'sent'
        ).count()
        
        failed = self.db.query(WhatsAppMessage).filter(
            WhatsAppMessage.business_id.in_(business_ids),
            WhatsAppMessage.status == 'failed'
        ).count()
        
        return {
            'total': total,
            'sent': sent,
            'failed': failed,
            'success_rate': (sent / total * 100) if total > 0 else 0
        }
    
    def __del__(self):
        """إغلاق الاتصال بقاعدة البيانات"""
        if hasattr(self, 'db'):
            self.db.close()


# دالة مساعدة لإنشاء رسائل بالجملة من الأعمال
def prepare_bulk_messages(businesses: List[Dict], message_template: str) -> List[Dict]:
    """
    تحضير رسائل متعددة من قائمة الأعمال
    
    Args:
        businesses: قائمة الأعمال
        message_template: قالب الرسالة مع متغيرات
        
    Returns:
        قائمة جاهزة للإرسال
    """
    messages = []
    
    for business in businesses:
        phone = business.get('phone')
        if not phone:
            continue
        
        # تخصيص الرسالة
        try:
            message = message_template.format(
                name=business.get('name', 'عزيزي العميل'),
                category=business.get('category', ''),
                address=business.get('address', ''),
            )
        except:
            message = message_template
        
        messages.append({
            'phone': phone,
            'message': message,
            'business_id': business.get('id')
        })
    
    return messages

