"""
سكريبت اختبار بسيط للتأكد من صحة الإعداد
"""

import sys

def test_imports():
    """اختبار استيراد المكتبات الأساسية"""
    print("🧪 اختبار المكتبات...")
    
    try:
        import flask
        print("✅ Flask")
    except ImportError:
        print("❌ Flask غير مثبت")
        return False
    
    try:
        import googlemaps
        print("✅ Google Maps")
    except ImportError:
        print("❌ Google Maps غير مثبت")
        return False
    
    try:
        import openai
        print("✅ OpenAI")
    except ImportError:
        print("❌ OpenAI غير مثبت")
        return False
    
    try:
        import sqlalchemy
        print("✅ SQLAlchemy")
    except ImportError:
        print("❌ SQLAlchemy غير مثبت")
        return False
    
    try:
        from dotenv import load_dotenv
        print("✅ Python Dotenv")
    except ImportError:
        print("❌ Python Dotenv غير مثبت")
        return False
    
    return True

def test_env_file():
    """اختبار ملف .env"""
    print("\n🧪 اختبار ملف .env...")
    
    import os
    from dotenv import load_dotenv
    
    if not os.path.exists('.env'):
        print("❌ ملف .env غير موجود")
        return False
    
    print("✅ ملف .env موجود")
    
    load_dotenv()
    
    keys = [
        'GOOGLE_MAPS_API_KEY',
        'OPENAI_API_KEY',
        'DEEPSEEK_API_KEY'
    ]
    
    all_ok = True
    for key in keys:
        value = os.getenv(key)
        if value:
            print(f"✅ {key} موجود")
        else:
            print(f"⚠️  {key} غير موجود أو فارغ")
            all_ok = False
    
    return all_ok

def test_config():
    """اختبار إعدادات التطبيق"""
    print("\n🧪 اختبار الإعدادات...")
    
    try:
        from config import Config
        Config.validate()
        print("✅ الإعدادات صحيحة")
        return True
    except Exception as e:
        print(f"❌ خطأ في الإعدادات: {str(e)}")
        return False

def test_database():
    """اختبار قاعدة البيانات"""
    print("\n🧪 اختبار قاعدة البيانات...")
    
    try:
        from database import init_db, SessionLocal, Business
        
        # إنشاء الجداول
        init_db()
        print("✅ تم إنشاء قاعدة البيانات")
        
        # اختبار الاتصال
        db = SessionLocal()
        count = db.query(Business).count()
        db.close()
        print(f"✅ الاتصال بقاعدة البيانات ناجح (عدد الأعمال: {count})")
        
        return True
    except Exception as e:
        print(f"❌ خطأ في قاعدة البيانات: {str(e)}")
        return False

def main():
    """الاختبار الرئيسي"""
    print("=" * 60)
    print("🔍 اختبار نظام التسويق الاحترافي")
    print("=" * 60)
    print()
    
    tests = [
        ("المكتبات", test_imports),
        ("ملف البيئة", test_env_file),
        ("الإعدادات", test_config),
        ("قاعدة البيانات", test_database)
    ]
    
    results = []
    
    for name, test_func in tests:
        try:
            result = test_func()
            results.append((name, result))
        except Exception as e:
            print(f"❌ خطأ في اختبار {name}: {str(e)}")
            results.append((name, False))
        print()
    
    # النتيجة النهائية
    print("=" * 60)
    print("📊 النتيجة النهائية:")
    print("=" * 60)
    
    passed = sum(1 for _, result in results if result)
    total = len(results)
    
    for name, result in results:
        status = "✅ نجح" if result else "❌ فشل"
        print(f"{name}: {status}")
    
    print()
    print(f"الإجمالي: {passed}/{total}")
    
    if passed == total:
        print("\n🎉 جميع الاختبارات نجحت! النظام جاهز للاستخدام")
        print("📝 لتشغيل النظام، استخدم: python app.py")
        return 0
    else:
        print("\n⚠️  بعض الاختبارات فشلت. يرجى التحقق من الأخطاء أعلاه")
        print("📝 راجع ملف README.md للحصول على المساعدة")
        return 1

if __name__ == "__main__":
    sys.exit(main())



