"""Test Google Maps API"""

import googlemaps
from dotenv import load_dotenv
import os

load_dotenv()

print("="*60)
print("Testing Google Maps API")
print("="*60)

api_key = os.getenv('GOOGLE_MAPS_API_KEY')

if not api_key:
    print("ERROR: No API Key found")
    exit(1)

print(f"API Key found: {api_key[:30]}...")

try:
    gmaps = googlemaps.Client(key=api_key)
    print("Client created successfully")
except Exception as e:
    print(f"ERROR creating client: {str(e)}")
    exit(1)

print("\nTesting search: restaurants in Riyadh")
print("="*60)

try:
    places_result = gmaps.places(
        query="restaurants in Riyadh",
        language='ar'
    )
    
    status = places_result.get('status')
    print(f"Status: {status}")
    
    if status == 'OK':
        results = places_result.get('results', [])
        print(f"\nSUCCESS! Found {len(results)} results")
        
        print("\nFirst 3 results:")
        for i, place in enumerate(results[:3], 1):
            print(f"\n{i}. {place.get('name', 'N/A')}")
            print(f"   Address: {place.get('formatted_address', 'N/A')[:50]}...")
            print(f"   Rating: {place.get('rating', 'N/A')}")
        
        print("\n" + "="*60)
        print("Google Maps API is WORKING!")
        print("="*60)
        
    elif status == 'REQUEST_DENIED':
        print("\nERROR: Request Denied")
        print(f"Reason: {places_result.get('error_message', 'Unknown')}")
        print("\nPossible solutions:")
        print("1. Enable Places API in Google Cloud Console")
        print("2. Check API Key validity")
        print("3. Add billing account (required even for free tier)")
        
    elif status == 'OVER_QUERY_LIMIT':
        print("\nERROR: Over Query Limit")
        print("Check your usage limits in Google Cloud Console")
        
    else:
        print(f"\nUnexpected status: {status}")
        print(f"Details: {places_result}")
        
except Exception as e:
    print(f"\nERROR: {str(e)}")
    import traceback
    traceback.print_exc()



