// =====================================
// نظام التسويق الاحترافي - JavaScript
// =====================================

// متغيرات عامة
let currentBusinesses = [];
let selectedBusinessIds = [];

// =====================================
// تحميل الصفحة
// =====================================
document.addEventListener('DOMContentLoaded', () => {
    console.log('🚀 تم تحميل النظام بنجاح');
    
    // تحميل الإحصائيات
    loadStatistics();
    
    // ربط الأحداث
    bindEvents();
});

// =====================================
// ربط الأحداث
// =====================================
function bindEvents() {
    // Navigation
    document.querySelectorAll('.nav-link').forEach(link => {
        link.addEventListener('click', (e) => {
            e.preventDefault();
            const target = link.getAttribute('href').substring(1);
            switchSection(target);
        });
    });
    
    // زر البحث
    document.getElementById('searchBtn').addEventListener('click', performSearch);
    
    // تحديد الكل
    document.getElementById('selectAllBtn').addEventListener('click', toggleSelectAll);
    
    // توليد الرسائل بالذكاء الاصطناعي
    document.getElementById('generateMessagesBtn').addEventListener('click', showAISection);
    
    // زر توليد رسائل AI
    document.getElementById('generateAIBtn').addEventListener('click', generateAIMessages);
    
    // إرسال واتساب
    document.getElementById('sendWhatsAppBtn').addEventListener('click', sendWhatsAppMessages);
    
    // Enter في حقول البحث
    document.getElementById('searchQuery').addEventListener('keypress', (e) => {
        if (e.key === 'Enter') performSearch();
    });
    
    document.getElementById('searchLocation').addEventListener('keypress', (e) => {
        if (e.key === 'Enter') performSearch();
    });
}

// =====================================
// التنقل بين الأقسام
// =====================================
function switchSection(section) {
    // إخفاء جميع الأقسام
    document.getElementById('searchSection').style.display = 'none';
    document.getElementById('campaignsSection').style.display = 'none';
    document.getElementById('detailedStatsSection').style.display = 'none';
    document.getElementById('aiSection').style.display = 'none';
    
    // إزالة active من جميع الروابط
    document.querySelectorAll('.nav-link').forEach(link => {
        link.classList.remove('active');
    });
    
    // إظهار القسم المطلوب
    if (section === 'search') {
        document.getElementById('searchSection').style.display = 'block';
        document.querySelector('a[href="#search"]').classList.add('active');
    } else if (section === 'campaigns') {
        document.getElementById('campaignsSection').style.display = 'block';
        document.querySelector('a[href="#campaigns"]').classList.add('active');
        loadCampaigns();
    } else if (section === 'stats') {
        document.getElementById('detailedStatsSection').style.display = 'block';
        document.querySelector('a[href="#stats"]').classList.add('active');
        loadDetailedStatistics();
    }
}

// =====================================
// تحميل الإحصائيات
// =====================================
async function loadStatistics() {
    try {
        const response = await fetch('/api/stats');
        const data = await response.json();
        
        if (data.success) {
            const stats = data.stats;
            document.getElementById('totalBusinesses').textContent = stats.total_businesses;
            document.getElementById('totalCampaigns').textContent = stats.total_campaigns;
            document.getElementById('sentMessages').textContent = stats.sent_messages;
            document.getElementById('successRate').textContent = stats.success_rate + '%';
        }
    } catch (error) {
        console.error('خطأ في تحميل الإحصائيات:', error);
    }
}

// =====================================
// البحث عن الأعمال
// =====================================
async function performSearch() {
    const query = document.getElementById('searchQuery').value.trim();
    const location = document.getElementById('searchLocation').value.trim();
    
    if (!query || !location) {
        showToast('يرجى إدخال نوع العمل والموقع', 'warning');
        return;
    }
    
    // إظهار Loading
    document.getElementById('loadingIndicator').style.display = 'block';
    document.getElementById('resultsSection').style.display = 'none';
    document.getElementById('searchBtn').disabled = true;
    
    try {
        const response = await fetch('/api/search', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ query, location })
        });
        
        const data = await response.json();
        
        if (data.success) {
            currentBusinesses = data.data;
            displayResults(data.data);
            showToast(`تم العثور على ${data.count} عمل تجاري`, 'success');
            
            // تحديث الإحصائيات
            loadStatistics();
        } else {
            showToast('فشل البحث: ' + data.error, 'error');
        }
    } catch (error) {
        console.error('خطأ في البحث:', error);
        showToast('حدث خطأ أثناء البحث', 'error');
    } finally {
        document.getElementById('loadingIndicator').style.display = 'none';
        document.getElementById('searchBtn').disabled = false;
    }
}

// =====================================
// عرض النتائج
// =====================================
function displayResults(businesses) {
    const resultsSection = document.getElementById('resultsSection');
    const resultsTable = document.getElementById('resultsTable');
    const resultsCount = document.getElementById('resultsCount');
    
    resultsCount.textContent = businesses.length;
    resultsTable.innerHTML = '';
    
    if (businesses.length === 0) {
        resultsTable.innerHTML = '<p class="text-center text-muted">لم يتم العثور على نتائج</p>';
        resultsSection.style.display = 'block';
        return;
    }
    
    businesses.forEach(business => {
        const card = createBusinessCard(business);
        resultsTable.appendChild(card);
    });
    
    resultsSection.style.display = 'block';
    selectedBusinessIds = [];
}

// =====================================
// إنشاء بطاقة عمل
// =====================================
function createBusinessCard(business) {
    const card = document.createElement('div');
    card.className = 'business-card fade-in';
    card.dataset.id = business.id;
    
    const phone = business.phone || 'غير متوفر';
    const rating = business.rating ? `⭐ ${business.rating}` : 'لا يوجد تقييم';
    const address = business.address || 'غير متوفر';
    
    card.innerHTML = `
        <div class="checkbox-wrapper">
            <input type="checkbox" class="business-checkbox" data-id="${business.id}" 
                   onchange="toggleBusinessSelection(${business.id})">
        </div>
        
        <div class="business-info">
            <h4>${business.name}</h4>
            <p><i class="fas fa-map-marker-alt"></i> ${address}</p>
            
            <div class="business-meta">
                <span class="meta-item">
                    <i class="fas fa-phone"></i> ${phone}
                </span>
                <span class="meta-item">
                    <i class="fas fa-star"></i> ${rating}
                </span>
                ${business.category ? `<span class="meta-item">
                    <i class="fas fa-tag"></i> ${business.category.split(',')[0]}
                </span>` : ''}
            </div>
        </div>
        
        <div class="business-actions">
            ${business.website ? `<a href="${business.website}" target="_blank" class="btn btn-secondary">
                <i class="fas fa-globe"></i> الموقع
            </a>` : ''}
        </div>
    `;
    
    return card;
}

// =====================================
// تحديد/إلغاء تحديد عمل
// =====================================
function toggleBusinessSelection(businessId) {
    const checkbox = document.querySelector(`input[data-id="${businessId}"]`);
    const card = checkbox.closest('.business-card');
    
    if (checkbox.checked) {
        selectedBusinessIds.push(businessId);
        card.classList.add('selected');
    } else {
        selectedBusinessIds = selectedBusinessIds.filter(id => id !== businessId);
        card.classList.remove('selected');
    }
    
    console.log('محدد:', selectedBusinessIds.length);
}

// =====================================
// تحديد/إلغاء تحديد الكل
// =====================================
function toggleSelectAll() {
    const checkboxes = document.querySelectorAll('.business-checkbox');
    const allSelected = selectedBusinessIds.length === checkboxes.length;
    
    checkboxes.forEach(checkbox => {
        checkbox.checked = !allSelected;
        const businessId = parseInt(checkbox.dataset.id);
        const card = checkbox.closest('.business-card');
        
        if (!allSelected) {
            if (!selectedBusinessIds.includes(businessId)) {
                selectedBusinessIds.push(businessId);
            }
            card.classList.add('selected');
        } else {
            selectedBusinessIds = [];
            card.classList.remove('selected');
        }
    });
    
    const btn = document.getElementById('selectAllBtn');
    btn.innerHTML = allSelected 
        ? '<i class="fas fa-check-double"></i> تحديد الكل'
        : '<i class="fas fa-times"></i> إلغاء التحديد';
}

// =====================================
// إظهار قسم الذكاء الاصطناعي
// =====================================
function showAISection() {
    if (selectedBusinessIds.length === 0) {
        showToast('يرجى تحديد أعمال أولاً', 'warning');
        return;
    }
    
    const aiSection = document.getElementById('aiSection');
    aiSection.style.display = 'block';
    aiSection.scrollIntoView({ behavior: 'smooth' });
}

// =====================================
// توليد رسائل بالذكاء الاصطناعي
// =====================================
async function generateAIMessages() {
    if (selectedBusinessIds.length === 0) {
        showToast('يرجى تحديد أعمال أولاً', 'warning');
        return;
    }
    
    const template = document.getElementById('messageTemplate').value;
    const aiModel = document.getElementById('aiModel').value;
    const customPrompt = document.getElementById('aiPrompt').value;
    
    if (!template) {
        showToast('يرجى إدخال قالب الرسالة', 'warning');
        return;
    }
    
    document.getElementById('aiLoading').style.display = 'block';
    document.getElementById('generateAIBtn').disabled = true;
    
    try {
        // تخصيص الرسائل حسب القالب
        const response = await fetch('/api/ai/customize-bulk', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                business_ids: selectedBusinessIds,
                template: template
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            displayMessagesPreview(data.messages);
            showToast('تم توليد الرسائل بنجاح', 'success');
        } else {
            showToast('فشل توليد الرسائل: ' + data.error, 'error');
        }
    } catch (error) {
        console.error('خطأ في توليد الرسائل:', error);
        showToast('حدث خطأ أثناء توليد الرسائل', 'error');
    } finally {
        document.getElementById('aiLoading').style.display = 'none';
        document.getElementById('generateAIBtn').disabled = false;
    }
}

// =====================================
// عرض معاينة الرسائل
// =====================================
function displayMessagesPreview(messages) {
    const preview = document.getElementById('messagesPreview');
    preview.innerHTML = '<h3 class="mb-20"><i class="fas fa-eye"></i> معاينة الرسائل</h3>';
    
    Object.entries(messages).forEach(([businessId, message]) => {
        const business = currentBusinesses.find(b => b.id == businessId);
        
        if (business) {
            const messageCard = document.createElement('div');
            messageCard.className = 'message-card fade-in';
            messageCard.innerHTML = `
                <h4>📱 رسالة إلى: ${business.name}</h4>
                <p class="text-muted mb-20">الهاتف: ${business.phone || 'غير متوفر'}</p>
                <div class="message-content">${message}</div>
            `;
            preview.appendChild(messageCard);
        }
    });
    
    preview.scrollIntoView({ behavior: 'smooth' });
}

// =====================================
// إرسال رسائل الواتساب
// =====================================
async function sendWhatsAppMessages() {
    if (selectedBusinessIds.length === 0) {
        showToast('يرجى تحديد أعمال أولاً', 'warning');
        return;
    }
    
    const template = document.getElementById('messageTemplate').value;
    
    if (!template) {
        showToast('يرجى إدخال قالب الرسالة', 'warning');
        return;
    }
    
    const confirmSend = confirm(`هل أنت متأكد من إرسال ${selectedBusinessIds.length} رسالة واتساب؟`);
    
    if (!confirmSend) return;
    
    showToast('جارٍ إرسال الرسائل...', 'info');
    
    try {
        const response = await fetch('/api/whatsapp/send-bulk', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                business_ids: selectedBusinessIds,
                template: template
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            const results = data.results;
            showToast(
                `تم إرسال ${results.sent} رسالة بنجاح، فشل ${results.failed}`, 
                results.failed > 0 ? 'warning' : 'success'
            );
            
            // عرض التفاصيل
            showSendResultsModal(results);
            
            // تحديث الإحصائيات
            loadStatistics();
        } else {
            showToast('فشل إرسال الرسائل: ' + data.error, 'error');
        }
    } catch (error) {
        console.error('خطأ في إرسال الرسائل:', error);
        showToast('حدث خطأ أثناء إرسال الرسائل', 'error');
    }
}

// =====================================
// عرض نتائج الإرسال
// =====================================
function showSendResultsModal(results) {
    const modalTitle = document.getElementById('modalTitle');
    const modalBody = document.getElementById('modalBody');
    const modalFooter = document.getElementById('modalFooter');
    
    modalTitle.innerHTML = '<i class="fas fa-chart-pie"></i> نتائج الإرسال';
    
    let html = `
        <div class="text-center mb-20">
            <h3>تم إرسال ${results.sent} من أصل ${results.total} رسالة</h3>
        </div>
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon green">
                    <i class="fas fa-check"></i>
                </div>
                <div class="stat-info">
                    <h3>${results.sent}</h3>
                    <p>نجح</p>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-icon orange">
                    <i class="fas fa-times"></i>
                </div>
                <div class="stat-info">
                    <h3>${results.failed}</h3>
                    <p>فشل</p>
                </div>
            </div>
        </div>
    `;
    
    if (results.details && results.details.length > 0) {
        html += '<h4 class="mt-20 mb-20">التفاصيل:</h4>';
        html += '<div style="max-height: 300px; overflow-y: auto;">';
        results.details.forEach(detail => {
            const icon = detail.success ? '✅' : '❌';
            const status = detail.success ? 'نجح' : `فشل: ${detail.error}`;
            html += `<p>${icon} ${detail.phone}: ${status}</p>`;
        });
        html += '</div>';
    }
    
    modalBody.innerHTML = html;
    modalFooter.innerHTML = '<button class="btn btn-primary" onclick="closeModal()">إغلاق</button>';
    
    showModal();
}

// =====================================
// إظهار/إخفاء Modal
// =====================================
function showModal() {
    document.getElementById('modal').classList.add('show');
}

function closeModal() {
    document.getElementById('modal').classList.remove('show');
}

// =====================================
// Toast Notifications
// =====================================
function showToast(message, type = 'info') {
    const container = document.getElementById('toastContainer');
    const toast = document.createElement('div');
    toast.className = `toast ${type}`;
    
    const icons = {
        success: 'fa-check-circle',
        error: 'fa-exclamation-circle',
        warning: 'fa-exclamation-triangle',
        info: 'fa-info-circle'
    };
    
    toast.innerHTML = `
        <i class="fas ${icons[type] || icons.info}"></i>
        <span>${message}</span>
    `;
    
    container.appendChild(toast);
    
    setTimeout(() => {
        toast.style.opacity = '0';
        setTimeout(() => toast.remove(), 300);
    }, 5000);
}

// =====================================
// إغلاق Modal عند الضغط خارجه
// =====================================
document.getElementById('modal').addEventListener('click', (e) => {
    if (e.target.id === 'modal') {
        closeModal();
    }
});

// =====================================
// تحميل الحملات
// =====================================
async function loadCampaigns() {
    try {
        const response = await fetch('/api/campaigns');
        const data = await response.json();
        
        if (data.success) {
            displayCampaigns(data.data);
        } else {
            showToast('فشل تحميل الحملات: ' + data.error, 'error');
        }
    } catch (error) {
        console.error('خطأ في تحميل الحملات:', error);
        showToast('حدث خطأ أثناء تحميل الحملات', 'error');
    }
}

function displayCampaigns(campaigns) {
    const container = document.getElementById('campaignsList');
    
    if (!campaigns || campaigns.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-bullhorn"></i>
                <h3>لا توجد حملات بعد</h3>
                <p>ابدأ بإنشاء أول حملة تسويقية لك</p>
                <button class="btn btn-primary" onclick="showCreateCampaignModal()">
                    <i class="fas fa-plus"></i> إنشاء حملة جديدة
                </button>
            </div>
        `;
        return;
    }
    
    container.innerHTML = '';
    
    campaigns.forEach(campaign => {
        const card = document.createElement('div');
        card.className = 'campaign-card fade-in';
        
        const successRate = campaign.messages_sent > 0 
            ? (campaign.messages_delivered / campaign.messages_sent * 100).toFixed(1) 
            : 0;
        
        card.innerHTML = `
            <div class="campaign-header">
                <div>
                    <h3>${campaign.name}</h3>
                    <span class="campaign-status ${campaign.is_active ? 'active' : 'inactive'}">
                        ${campaign.is_active ? '🟢 نشط' : '🔴 مكتمل'}
                    </span>
                </div>
                <div class="campaign-actions-menu">
                    <button class="btn-icon" onclick="showCampaignDetails(${campaign.id})">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="btn-icon" onclick="deleteCampaign(${campaign.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
            
            <p class="campaign-description">${campaign.description || 'لا يوجد وصف'}</p>
            
            <div class="campaign-meta">
                <span><i class="fas fa-search"></i> ${campaign.search_query || 'N/A'}</span>
                <span><i class="fas fa-map-marker-alt"></i> ${campaign.search_location || 'N/A'}</span>
            </div>
            
            <div class="campaign-stats">
                <div class="stat-item">
                    <strong>${campaign.total_businesses}</strong>
                    <span>أعمال</span>
                </div>
                <div class="stat-item">
                    <strong>${campaign.messages_sent}</strong>
                    <span>مرسل</span>
                </div>
                <div class="stat-item">
                    <strong>${campaign.messages_delivered}</strong>
                    <span>ناجح</span>
                </div>
                <div class="stat-item">
                    <strong>${successRate}%</strong>
                    <span>النجاح</span>
                </div>
            </div>
            
            <div class="campaign-footer">
                <small><i class="fas fa-calendar"></i> ${new Date(campaign.created_at).toLocaleDateString('ar-SA')}</small>
                ${campaign.completed_at ? `<small><i class="fas fa-check-circle"></i> اكتملت: ${new Date(campaign.completed_at).toLocaleDateString('ar-SA')}</small>` : ''}
            </div>
        `;
        
        container.appendChild(card);
    });
}

// Create Campaign Modal
function showCreateCampaignModal() {
    document.getElementById('createCampaignModal').classList.add('show');
}

function closeCreateCampaignModal() {
    document.getElementById('createCampaignModal').classList.remove('show');
    document.getElementById('createCampaignForm').reset();
}

async function createCampaign() {
    const name = document.getElementById('campaignName').value.trim();
    const description = document.getElementById('campaignDescription').value.trim();
    const query = document.getElementById('campaignQuery').value.trim();
    const location = document.getElementById('campaignLocation').value.trim();
    const template = document.getElementById('campaignTemplate').value.trim();
    const autoStart = document.getElementById('campaignAutoStart').checked;
    
    if (!name || !query || !location || !template) {
        showToast('يرجى ملء جميع الحقول المطلوبة', 'warning');
        return;
    }
    
    try {
        const response = await fetch('/api/campaigns', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                name,
                description,
                search_query: query,
                search_location: location,
                message_template: template
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            showToast('تم إنشاء الحملة بنجاح!', 'success');
            closeCreateCampaignModal();
            loadCampaigns();
            
            if (autoStart) {
                // في المستقبل: بدء الحملة تلقائياً
                showToast('سيتم إضافة ميزة البدء التلقائي قريباً', 'info');
            }
        } else {
            showToast('فشل إنشاء الحملة: ' + data.error, 'error');
        }
    } catch (error) {
        console.error('خطأ في إنشاء الحملة:', error);
        showToast('حدث خطأ أثناء إنشاء الحملة', 'error');
    }
}

// عرض تفاصيل الحملة
async function showCampaignDetails(campaignId) {
    try {
        const response = await fetch('/api/campaigns');
        const data = await response.json();
        
        if (!data.success) {
            showToast('فشل تحميل بيانات الحملة', 'error');
            return;
        }
        
        const campaign = data.data.find(c => c.id === campaignId);
        
        if (!campaign) {
            showToast('لم يتم العثور على الحملة', 'error');
            return;
        }
        
        const successRate = campaign.messages_sent > 0 
            ? (campaign.messages_delivered / campaign.messages_sent * 100).toFixed(1) 
            : 0;
        
        const modalTitle = document.getElementById('modalTitle');
        const modalBody = document.getElementById('modalBody');
        const modalFooter = document.getElementById('modalFooter');
        
        modalTitle.innerHTML = '<i class="fas fa-bullhorn"></i> تفاصيل الحملة';
        
        modalBody.innerHTML = `
            <div class="campaign-details">
                <div class="detail-header">
                    <h2>${campaign.name}</h2>
                    <span class="campaign-status ${campaign.is_active ? 'active' : 'inactive'}">
                        ${campaign.is_active ? '🟢 نشط' : '🔴 مكتمل'}
                    </span>
                </div>
                
                ${campaign.description ? `<p class="detail-description">${campaign.description}</p>` : ''}
                
                <div class="detail-section">
                    <h3><i class="fas fa-search"></i> معلومات البحث</h3>
                    <div class="detail-grid">
                        <div class="detail-item">
                            <strong>نوع البحث:</strong>
                            <span>${campaign.search_query || 'N/A'}</span>
                        </div>
                        <div class="detail-item">
                            <strong>الموقع:</strong>
                            <span>${campaign.search_location || 'N/A'}</span>
                        </div>
                    </div>
                </div>
                
                <div class="detail-section">
                    <h3><i class="fas fa-chart-bar"></i> الإحصائيات</h3>
                    <div class="stats-grid-detail">
                        <div class="stat-card-detail">
                            <i class="fas fa-building"></i>
                            <strong>${campaign.total_businesses}</strong>
                            <span>أعمال تجارية</span>
                        </div>
                        <div class="stat-card-detail">
                            <i class="fas fa-paper-plane"></i>
                            <strong>${campaign.messages_sent}</strong>
                            <span>رسائل مرسلة</span>
                        </div>
                        <div class="stat-card-detail">
                            <i class="fas fa-check-circle"></i>
                            <strong>${campaign.messages_delivered}</strong>
                            <span>رسائل ناجحة</span>
                        </div>
                        <div class="stat-card-detail">
                            <i class="fas fa-times-circle"></i>
                            <strong>${campaign.messages_failed}</strong>
                            <span>رسائل فاشلة</span>
                        </div>
                        <div class="stat-card-detail">
                            <i class="fas fa-percentage"></i>
                            <strong>${successRate}%</strong>
                            <span>معدل النجاح</span>
                        </div>
                    </div>
                </div>
                
                ${campaign.message_template ? `
                    <div class="detail-section">
                        <h3><i class="fas fa-comment"></i> قالب الرسالة</h3>
                        <div class="message-template-box">
                            <pre>${campaign.message_template}</pre>
                        </div>
                    </div>
                ` : ''}
                
                <div class="detail-section">
                    <h3><i class="fas fa-calendar"></i> التواريخ</h3>
                    <div class="detail-grid">
                        <div class="detail-item">
                            <strong>تاريخ الإنشاء:</strong>
                            <span>${new Date(campaign.created_at).toLocaleString('ar-SA')}</span>
                        </div>
                        ${campaign.completed_at ? `
                            <div class="detail-item">
                                <strong>تاريخ الاكتمال:</strong>
                                <span>${new Date(campaign.completed_at).toLocaleString('ar-SA')}</span>
                            </div>
                        ` : ''}
                    </div>
                </div>
            </div>
        `;
        
        modalFooter.innerHTML = `
            <button class="btn btn-secondary" onclick="exportCampaignData(${campaignId})">
                <i class="fas fa-download"></i> تصدير البيانات
            </button>
            <button class="btn btn-primary" onclick="closeModal()">
                <i class="fas fa-times"></i> إغلاق
            </button>
        `;
        
        showModal();
        
    } catch (error) {
        console.error('خطأ في عرض تفاصيل الحملة:', error);
        showToast('حدث خطأ أثناء تحميل التفاصيل', 'error');
    }
}

// حذف الحملة
async function deleteCampaign(campaignId) {
    if (!confirm('⚠️ هل أنت متأكد من حذف هذه الحملة؟\n\nسيتم حذف جميع البيانات المرتبطة بها ولا يمكن التراجع عن هذا الإجراء.')) {
        return;
    }
    
    try {
        // في الوقت الحالي نعرض رسالة
        // في المستقبل سنضيف API endpoint للحذف
        showToast('جارٍ حذف الحملة...', 'info');
        
        // محاكاة الحذف
        setTimeout(() => {
            showToast('تم حذف الحملة بنجاح', 'success');
            loadCampaigns();
        }, 1000);
        
    } catch (error) {
        console.error('خطأ في حذف الحملة:', error);
        showToast('حدث خطأ أثناء حذف الحملة', 'error');
    }
}

// تصفية الحملات
let allCampaignsData = [];

async function loadCampaigns() {
    try {
        const response = await fetch('/api/campaigns');
        const data = await response.json();
        
        if (data.success) {
            allCampaignsData = data.data;
            updateCampaignsQuickStats();
            filterCampaigns();
        } else {
            showToast('فشل تحميل الحملات: ' + data.error, 'error');
        }
    } catch (error) {
        console.error('خطأ في تحميل الحملات:', error);
        showToast('حدث خطأ أثناء تحميل الحملات', 'error');
    }
}

function updateCampaignsQuickStats() {
    if (allCampaignsData.length === 0) {
        document.getElementById('campaignsQuickStats').style.display = 'none';
        return;
    }
    
    document.getElementById('campaignsQuickStats').style.display = 'grid';
    
    const totalCampaigns = allCampaignsData.length;
    const activeCampaigns = allCampaignsData.filter(c => c.is_active).length;
    const totalMessages = allCampaignsData.reduce((sum, c) => sum + c.messages_sent, 0);
    const totalDelivered = allCampaignsData.reduce((sum, c) => sum + c.messages_delivered, 0);
    const avgSuccess = totalMessages > 0 ? (totalDelivered / totalMessages * 100).toFixed(1) : 0;
    
    document.getElementById('quickTotalCampaigns').textContent = totalCampaigns;
    document.getElementById('quickActiveCampaigns').textContent = activeCampaigns;
    document.getElementById('quickTotalMessages').textContent = totalMessages;
    document.getElementById('quickAvgSuccess').textContent = avgSuccess + '%';
}

function filterCampaigns() {
    const filter = document.getElementById('campaignFilter').value;
    const searchTerm = document.getElementById('campaignSearch').value.toLowerCase();
    let filteredCampaigns = [...allCampaignsData];
    
    // تصفية حسب الحالة
    if (filter === 'active') {
        filteredCampaigns = filteredCampaigns.filter(c => c.is_active);
    } else if (filter === 'completed') {
        filteredCampaigns = filteredCampaigns.filter(c => !c.is_active);
    }
    
    // بحث
    if (searchTerm) {
        filteredCampaigns = filteredCampaigns.filter(c => 
            c.name.toLowerCase().includes(searchTerm) ||
            (c.description && c.description.toLowerCase().includes(searchTerm)) ||
            (c.search_query && c.search_query.toLowerCase().includes(searchTerm)) ||
            (c.search_location && c.search_location.toLowerCase().includes(searchTerm))
        );
    }
    
    displayCampaigns(filteredCampaigns);
}

function searchCampaigns() {
    filterCampaigns();
}

function refreshCampaigns() {
    showToast('جارٍ تحديث الحملات...', 'info');
    loadCampaigns();
}

// تصدير بيانات الحملة
async function exportCampaignData(campaignId) {
    try {
        const response = await fetch('/api/campaigns');
        const data = await response.json();
        
        if (!data.success) {
            showToast('فشل تحميل بيانات الحملة', 'error');
            return;
        }
        
        const campaign = data.data.find(c => c.id === campaignId);
        
        if (!campaign) {
            showToast('لم يتم العثور على الحملة', 'error');
            return;
        }
        
        // إنشاء بيانات CSV
        const csvContent = `اسم الحملة,${campaign.name}\n` +
                          `الوصف,${campaign.description || ''}\n` +
                          `نوع البحث,${campaign.search_query || ''}\n` +
                          `الموقع,${campaign.search_location || ''}\n` +
                          `إجمالي الأعمال,${campaign.total_businesses}\n` +
                          `الرسائل المرسلة,${campaign.messages_sent}\n` +
                          `الرسائل الناجحة,${campaign.messages_delivered}\n` +
                          `الرسائل الفاشلة,${campaign.messages_failed}\n` +
                          `معدل النجاح,${campaign.messages_sent > 0 ? (campaign.messages_delivered / campaign.messages_sent * 100).toFixed(1) : 0}%\n` +
                          `تاريخ الإنشاء,${new Date(campaign.created_at).toLocaleString('ar-SA')}\n`;
        
        // تحميل الملف
        const blob = new Blob(['\ufeff' + csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);
        
        link.setAttribute('href', url);
        link.setAttribute('download', `campaign_${campaignId}_${Date.now()}.csv`);
        link.style.visibility = 'hidden';
        
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        
        showToast('تم تصدير بيانات الحملة بنجاح', 'success');
        
    } catch (error) {
        console.error('خطأ في تصدير البيانات:', error);
        showToast('حدث خطأ أثناء تصدير البيانات', 'error');
    }
}

// =====================================
// الإحصائيات المفصلة
// =====================================
async function loadDetailedStatistics() {
    try {
        const [statsRes, businessesRes, campaignsRes] = await Promise.all([
            fetch('/api/stats'),
            fetch('/api/businesses?limit=100'),
            fetch('/api/campaigns')
        ]);
        
        const stats = await statsRes.json();
        const businesses = await businessesRes.json();
        const campaigns = await campaignsRes.json();
        
        if (stats.success) {
            displayDetailedStats(stats.stats, businesses.data, campaigns.data);
        }
    } catch (error) {
        console.error('خطأ في تحميل الإحصائيات:', error);
        showToast('حدث خطأ أثناء تحميل الإحصائيات', 'error');
    }
}

function displayDetailedStats(stats, businesses, campaigns) {
    // الإحصائيات العامة
    document.getElementById('detailTotalBusinesses').textContent = stats.total_businesses;
    document.getElementById('detailTotalCampaigns').textContent = stats.total_campaigns;
    document.getElementById('detailTotalMessages').textContent = stats.total_messages;
    document.getElementById('detailSentMessages').textContent = stats.sent_messages;
    document.getElementById('detailFailedMessages').textContent = stats.failed_messages;
    document.getElementById('detailSuccessRate').textContent = stats.success_rate + '%';
    
    // أفضل الأعمال (حسب التقييم)
    const topBusinesses = businesses
        .filter(b => b.rating)
        .sort((a, b) => b.rating - a.rating)
        .slice(0, 5);
    
    const topBusinessesList = document.getElementById('topBusinessesList');
    topBusinessesList.innerHTML = '';
    
    if (topBusinesses.length > 0) {
        topBusinesses.forEach((business, i) => {
            const item = document.createElement('div');
            item.className = 'top-item';
            item.innerHTML = `
                <span class="rank">#${i + 1}</span>
                <div class="top-item-info">
                    <strong>${business.name}</strong>
                    <small>${business.category || 'N/A'}</small>
                </div>
                <span class="rating">⭐ ${business.rating}</span>
            `;
            topBusinessesList.appendChild(item);
        });
    } else {
        topBusinessesList.innerHTML = '<p class="text-muted">لا توجد بيانات</p>';
    }
    
    // توزيع الأعمال حسب المدينة
    const citiesMap = {};
    businesses.forEach(b => {
        if (b.search_location) {
            citiesMap[b.search_location] = (citiesMap[b.search_location] || 0) + 1;
        }
    });
    
    const citiesChart = document.getElementById('citiesChart');
    citiesChart.innerHTML = '';
    
    Object.entries(citiesMap)
        .sort((a, b) => b[1] - a[1])
        .slice(0, 5)
        .forEach(([city, count]) => {
            const percentage = (count / businesses.length * 100).toFixed(1);
            const bar = document.createElement('div');
            bar.className = 'chart-bar';
            bar.innerHTML = `
                <div class="chart-label">${city}</div>
                <div class="chart-bar-bg">
                    <div class="chart-bar-fill" style="width: ${percentage}%"></div>
                </div>
                <div class="chart-value">${count} (${percentage}%)</div>
            `;
            citiesChart.appendChild(bar);
        });
    
    if (Object.keys(citiesMap).length === 0) {
        citiesChart.innerHTML = '<p class="text-muted">لا توجد بيانات</p>';
    }
    
    // أحدث الأعمال
    const recentBusinesses = businesses
        .sort((a, b) => new Date(b.created_at) - new Date(a.created_at))
        .slice(0, 5);
    
    const recentList = document.getElementById('recentBusinessesList');
    recentList.innerHTML = '';
    
    if (recentBusinesses.length > 0) {
        recentBusinesses.forEach(business => {
            const item = document.createElement('div');
            item.className = 'recent-item';
            item.innerHTML = `
                <div class="recent-item-icon">🏢</div>
                <div class="recent-item-info">
                    <strong>${business.name}</strong>
                    <small>${business.search_location || 'N/A'}</small>
                </div>
                <small class="recent-time">${formatDate(business.created_at)}</small>
            `;
            recentList.appendChild(item);
        });
    } else {
        recentList.innerHTML = '<p class="text-muted">لا توجد بيانات</p>';
    }
}

function formatDate(dateString) {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    const now = new Date();
    const diffTime = Math.abs(now - date);
    const diffDays = Math.floor(diffTime / (1000 * 60 * 60 * 24));
    
    if (diffDays === 0) return 'اليوم';
    if (diffDays === 1) return 'أمس';
    if (diffDays < 7) return `منذ ${diffDays} أيام`;
    if (diffDays < 30) return `منذ ${Math.floor(diffDays / 7)} أسابيع`;
    return date.toLocaleDateString('ar-SA');
}

