import googlemaps
import time
from typing import List, Dict, Optional
from config import Config
from database import SessionLocal, Business
import logging

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class GoogleMapsScraper:
    """استخراج بيانات الأعمال من خرائط جوجل"""
    
    def __init__(self):
        self.gmaps = googlemaps.Client(key=Config.GOOGLE_MAPS_API_KEY)
        self.db = SessionLocal()
    
    def search_places(self, query: str, location: str) -> List[Dict]:
        """
        البحث عن الأماكن في خرائط جوجل
        
        Args:
            query: نوع العمل (مثال: مطاعم، صيدليات، محلات)
            location: المنطقة أو المدينة
            
        Returns:
            قائمة بالأعمال المستخرجة
        """
        try:
            logger.info(f"🔍 البحث عن: {query} في {location}")
            
            # البحث الأولي
            search_query = f"{query} في {location}"
            places_result = self.gmaps.places(query=search_query, language='ar')
            
            businesses = []
            
            if places_result.get('status') == 'OK':
                results = places_result.get('results', [])
                logger.info(f"✅ تم العثور على {len(results)} نتيجة")
                
                for place in results:
                    business_data = self._extract_place_details(place, query, location)
                    if business_data:
                        businesses.append(business_data)
                        # حفظ في قاعدة البيانات
                        self._save_business(business_data)
                
                # جلب المزيد من النتائج إن وجدت
                while 'next_page_token' in places_result and len(businesses) < Config.MAX_RESULTS_PER_SEARCH:
                    time.sleep(2)  # الانتظار قبل طلب الصفحة التالية
                    places_result = self.gmaps.places(
                        query=search_query,
                        page_token=places_result['next_page_token'],
                        language='ar'
                    )
                    
                    if places_result.get('status') == 'OK':
                        for place in places_result.get('results', []):
                            if len(businesses) >= Config.MAX_RESULTS_PER_SEARCH:
                                break
                            business_data = self._extract_place_details(place, query, location)
                            if business_data:
                                businesses.append(business_data)
                                self._save_business(business_data)
            
            logger.info(f"✅ تم استخراج {len(businesses)} عمل بنجاح")
            return businesses
            
        except Exception as e:
            logger.error(f"❌ خطأ في استخراج البيانات: {str(e)}")
            raise
    
    def _extract_place_details(self, place: Dict, query: str, location: str) -> Optional[Dict]:
        """استخراج تفاصيل المكان"""
        try:
            place_id = place.get('place_id')
            if not place_id:
                return None
            
            # جلب التفاصيل الكاملة
            details = self.gmaps.place(
                place_id=place_id,
                fields=['name', 'formatted_address', 'formatted_phone_number', 
                       'international_phone_number', 'website', 'rating', 
                       'user_ratings_total', 'geometry', 'types'],
                language='ar'
            )
            
            if details.get('status') != 'OK':
                return None
            
            result = details.get('result', {})
            geometry = result.get('geometry', {})
            location_data = geometry.get('location', {})
            
            business = {
                'name': result.get('name', ''),
                'place_id': place_id,
                'address': result.get('formatted_address', ''),
                'phone': result.get('formatted_phone_number') or result.get('international_phone_number', ''),
                'website': result.get('website', ''),
                'rating': result.get('rating', 0.0),
                'total_ratings': result.get('user_ratings_total', 0),
                'category': ', '.join(result.get('types', [])[:3]),
                'location': {
                    'lat': location_data.get('lat'),
                    'lng': location_data.get('lng')
                },
                'search_query': query,
                'search_location': location
            }
            
            return business
            
        except Exception as e:
            logger.error(f"❌ خطأ في استخراج تفاصيل المكان: {str(e)}")
            return None
    
    def _save_business(self, business_data: Dict):
        """حفظ بيانات العمل في قاعدة البيانات"""
        try:
            # التحقق من عدم وجود المكان مسبقاً
            existing = self.db.query(Business).filter_by(
                place_id=business_data['place_id']
            ).first()
            
            if existing:
                # تحديث البيانات
                for key, value in business_data.items():
                    if key == 'location' and value:
                        existing.location_lat = value.get('lat')
                        existing.location_lng = value.get('lng')
                    elif key != 'location':
                        setattr(existing, key, value)
            else:
                # إضافة عمل جديد
                location = business_data.pop('location', {})
                business = Business(
                    **business_data,
                    location_lat=location.get('lat') if location else None,
                    location_lng=location.get('lng') if location else None
                )
                self.db.add(business)
            
            self.db.commit()
            
        except Exception as e:
            logger.error(f"❌ خطأ في حفظ البيانات: {str(e)}")
            self.db.rollback()
    
    def get_all_businesses(self, limit: int = 100) -> List[Dict]:
        """الحصول على جميع الأعمال المحفوظة"""
        businesses = self.db.query(Business).limit(limit).all()
        return [b.to_dict() for b in businesses]
    
    def __del__(self):
        """إغلاق الاتصال بقاعدة البيانات"""
        if hasattr(self, 'db'):
            self.db.close()



