"""
تصدير البيانات من قاعدة البيانات إلى ملفات Excel أو CSV
"""

import csv
import json
from datetime import datetime
from database import SessionLocal, Business, WhatsAppMessage, Campaign

def export_businesses_to_csv(filename='businesses_export.csv'):
    """تصدير الأعمال إلى CSV"""
    print(f"📊 تصدير الأعمال إلى {filename}...")
    
    db = SessionLocal()
    businesses = db.query(Business).all()
    
    if not businesses:
        print("⚠️  لا توجد بيانات للتصدير")
        db.close()
        return
    
    # كتابة CSV
    with open(filename, 'w', newline='', encoding='utf-8-sig') as f:
        writer = csv.writer(f)
        
        # العناوين
        writer.writerow([
            'الاسم',
            'الهاتف',
            'العنوان',
            'التقييم',
            'عدد التقييمات',
            'الفئة',
            'الموقع الإلكتروني',
            'خط العرض',
            'خط الطول',
            'نوع البحث',
            'موقع البحث',
            'تاريخ الإضافة'
        ])
        
        # البيانات
        for business in businesses:
            writer.writerow([
                business.name,
                business.phone or '',
                business.address or '',
                business.rating or '',
                business.total_ratings or '',
                business.category or '',
                business.website or '',
                business.location_lat or '',
                business.location_lng or '',
                business.search_query or '',
                business.search_location or '',
                business.created_at.strftime('%Y-%m-%d %H:%M:%S') if business.created_at else ''
            ])
    
    print(f"✅ تم تصدير {len(businesses)} عمل بنجاح")
    print(f"📁 الملف: {filename}")
    
    db.close()


def export_businesses_to_json(filename='businesses_export.json'):
    """تصدير الأعمال إلى JSON"""
    print(f"📊 تصدير الأعمال إلى {filename}...")
    
    db = SessionLocal()
    businesses = db.query(Business).all()
    
    if not businesses:
        print("⚠️  لا توجد بيانات للتصدير")
        db.close()
        return
    
    # تحويل إلى قاموس
    data = {
        'export_date': datetime.now().isoformat(),
        'total_businesses': len(businesses),
        'businesses': [b.to_dict() for b in businesses]
    }
    
    # كتابة JSON
    with open(filename, 'w', encoding='utf-8') as f:
        json.dump(data, f, ensure_ascii=False, indent=2)
    
    print(f"✅ تم تصدير {len(businesses)} عمل بنجاح")
    print(f"📁 الملف: {filename}")
    
    db.close()


def export_messages_to_csv(filename='messages_export.csv'):
    """تصدير رسائل WhatsApp إلى CSV"""
    print(f"📊 تصدير الرسائل إلى {filename}...")
    
    db = SessionLocal()
    messages = db.query(WhatsAppMessage).all()
    
    if not messages:
        print("⚠️  لا توجد رسائل للتصدير")
        db.close()
        return
    
    # كتابة CSV
    with open(filename, 'w', newline='', encoding='utf-8-sig') as f:
        writer = csv.writer(f)
        
        # العناوين
        writer.writerow([
            'رقم الهاتف',
            'الحالة',
            'تاريخ الإرسال',
            'الرسالة',
            'رسالة الخطأ'
        ])
        
        # البيانات
        for msg in messages:
            writer.writerow([
                msg.phone_number,
                msg.status,
                msg.sent_at.strftime('%Y-%m-%d %H:%M:%S') if msg.sent_at else '',
                msg.message_content[:100] + '...' if len(msg.message_content) > 100 else msg.message_content,
                msg.error_message or ''
            ])
    
    print(f"✅ تم تصدير {len(messages)} رسالة بنجاح")
    print(f"📁 الملف: {filename}")
    
    db.close()


def export_campaigns_report(filename='campaigns_report.csv'):
    """تصدير تقرير الحملات"""
    print(f"📊 تصدير تقرير الحملات إلى {filename}...")
    
    db = SessionLocal()
    campaigns = db.query(Campaign).all()
    
    if not campaigns:
        print("⚠️  لا توجد حملات للتصدير")
        db.close()
        return
    
    # كتابة CSV
    with open(filename, 'w', newline='', encoding='utf-8-sig') as f:
        writer = csv.writer(f)
        
        # العناوين
        writer.writerow([
            'اسم الحملة',
            'نوع البحث',
            'الموقع',
            'إجمالي الأعمال',
            'رسائل مرسلة',
            'رسائل ناجحة',
            'رسائل فاشلة',
            'معدل النجاح %',
            'تاريخ الإنشاء',
            'تاريخ الإكمال',
            'نشط'
        ])
        
        # البيانات
        for campaign in campaigns:
            success_rate = (campaign.messages_delivered / campaign.messages_sent * 100) if campaign.messages_sent > 0 else 0
            
            writer.writerow([
                campaign.name,
                campaign.search_query or '',
                campaign.search_location or '',
                campaign.total_businesses,
                campaign.messages_sent,
                campaign.messages_delivered,
                campaign.messages_failed,
                f"{success_rate:.1f}",
                campaign.created_at.strftime('%Y-%m-%d %H:%M:%S') if campaign.created_at else '',
                campaign.completed_at.strftime('%Y-%m-%d %H:%M:%S') if campaign.completed_at else '',
                'نعم' if campaign.is_active else 'لا'
            ])
    
    print(f"✅ تم تصدير {len(campaigns)} حملة بنجاح")
    print(f"📁 الملف: {filename}")
    
    db.close()


def export_statistics(filename='statistics_report.txt'):
    """تصدير تقرير إحصائيات شامل"""
    print(f"📊 تصدير الإحصائيات إلى {filename}...")
    
    db = SessionLocal()
    
    # جمع الإحصائيات
    total_businesses = db.query(Business).count()
    businesses_with_phone = db.query(Business).filter(Business.phone != None).count()
    high_rated = db.query(Business).filter(Business.rating >= 4.0).count()
    
    total_messages = db.query(WhatsAppMessage).count()
    sent_messages = db.query(WhatsAppMessage).filter_by(status='sent').count()
    failed_messages = db.query(WhatsAppMessage).filter_by(status='failed').count()
    
    total_campaigns = db.query(Campaign).count()
    active_campaigns = db.query(Campaign).filter_by(is_active=True).count()
    
    # كتابة التقرير
    with open(filename, 'w', encoding='utf-8') as f:
        f.write("╔═══════════════════════════════════════════════════════════╗\n")
        f.write("║                                                           ║\n")
        f.write("║           📊 تقرير إحصائيات النظام الشامل 📊             ║\n")
        f.write("║                                                           ║\n")
        f.write("╚═══════════════════════════════════════════════════════════╝\n\n")
        
        f.write(f"تاريخ التقرير: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n\n")
        
        f.write("=" * 60 + "\n")
        f.write("📊 الأعمال التجارية:\n")
        f.write("=" * 60 + "\n")
        f.write(f"إجمالي الأعمال المستخرجة: {total_businesses}\n")
        f.write(f"أعمال بأرقام هواتف: {businesses_with_phone}\n")
        f.write(f"أعمال بتقييم 4+ نجوم: {high_rated}\n")
        f.write(f"نسبة الأعمال بأرقام: {(businesses_with_phone/total_businesses*100):.1f}%\n" if total_businesses > 0 else "")
        f.write("\n")
        
        f.write("=" * 60 + "\n")
        f.write("📱 رسائل WhatsApp:\n")
        f.write("=" * 60 + "\n")
        f.write(f"إجمالي الرسائل: {total_messages}\n")
        f.write(f"رسائل مرسلة بنجاح: {sent_messages}\n")
        f.write(f"رسائل فاشلة: {failed_messages}\n")
        f.write(f"معدل النجاح: {(sent_messages/total_messages*100):.1f}%\n" if total_messages > 0 else "")
        f.write("\n")
        
        f.write("=" * 60 + "\n")
        f.write("📢 الحملات التسويقية:\n")
        f.write("=" * 60 + "\n")
        f.write(f"إجمالي الحملات: {total_campaigns}\n")
        f.write(f"حملات نشطة: {active_campaigns}\n")
        f.write(f"حملات مكتملة: {total_campaigns - active_campaigns}\n")
        f.write("\n")
        
        # أفضل الحملات
        best_campaigns = db.query(Campaign).order_by(
            Campaign.messages_delivered.desc()
        ).limit(5).all()
        
        if best_campaigns:
            f.write("=" * 60 + "\n")
            f.write("🏆 أفضل 5 حملات (حسب الرسائل الناجحة):\n")
            f.write("=" * 60 + "\n")
            for i, campaign in enumerate(best_campaigns, 1):
                f.write(f"{i}. {campaign.name}\n")
                f.write(f"   رسائل ناجحة: {campaign.messages_delivered}\n")
                f.write(f"   معدل النجاح: {(campaign.messages_delivered/campaign.messages_sent*100):.1f}%\n" if campaign.messages_sent > 0 else "")
                f.write("\n")
    
    print(f"✅ تم تصدير التقرير بنجاح")
    print(f"📁 الملف: {filename}")
    
    db.close()


def export_all():
    """تصدير جميع البيانات"""
    print("\n╔═══════════════════════════════════════════════════════════╗")
    print("║                                                           ║")
    print("║            📤 تصدير جميع بيانات النظام 📤                ║")
    print("║                                                           ║")
    print("╚═══════════════════════════════════════════════════════════╝\n")
    
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    
    export_businesses_to_csv(f'businesses_{timestamp}.csv')
    print()
    
    export_businesses_to_json(f'businesses_{timestamp}.json')
    print()
    
    export_messages_to_csv(f'messages_{timestamp}.csv')
    print()
    
    export_campaigns_report(f'campaigns_{timestamp}.csv')
    print()
    
    export_statistics(f'statistics_{timestamp}.txt')
    
    print("\n" + "=" * 60)
    print("✅ تم تصدير جميع البيانات بنجاح!")
    print("=" * 60)


if __name__ == "__main__":
    export_all()



