"""
أمثلة عملية لاستخدام النظام مباشرة من Python
يمكنك استخدام هذه الأمثلة لاستدعاء وظائف النظام بدون الواجهة
"""

from google_maps_scraper import GoogleMapsScraper
from ai_integration import AIAssistant
from whatsapp_integration import WhatsAppSender, prepare_bulk_messages
from database import init_db, SessionLocal, Business
import time

# تهيئة قاعدة البيانات
init_db()

def example_1_search_businesses():
    """مثال 1: البحث عن أعمال تجارية"""
    print("=" * 60)
    print("مثال 1: البحث في Google Maps")
    print("=" * 60)
    
    scraper = GoogleMapsScraper()
    
    # البحث عن مطاعم في الرياض
    businesses = scraper.search_places(
        query="مطاعم",
        location="الرياض"
    )
    
    print(f"\n✅ تم العثور على {len(businesses)} مطعم")
    
    # عرض أول 5 نتائج
    for i, business in enumerate(businesses[:5], 1):
        print(f"\n{i}. {business['name']}")
        print(f"   📞 {business.get('phone', 'غير متوفر')}")
        print(f"   ⭐ {business.get('rating', 'N/A')}")
    
    return businesses


def example_2_generate_ai_messages():
    """مثال 2: توليد رسائل بالذكاء الاصطناعي"""
    print("\n" + "=" * 60)
    print("مثال 2: توليد رسائل تسويقية بالذكاء الاصطناعي")
    print("=" * 60)
    
    # استخدام DeepSeek (أرخص)
    ai = AIAssistant(use_deepseek=True)
    
    # بيانات عمل تجريبي
    business = {
        'name': 'مطعم الواحة',
        'category': 'مطعم',
        'rating': 4.5
    }
    
    # توليد رسالة
    message = ai.generate_marketing_message(
        business_data=business,
        custom_prompt="اجعل الرسالة جذابة ومختصرة"
    )
    
    print(f"\n✅ الرسالة المولدة:")
    print("-" * 60)
    print(message)
    print("-" * 60)
    
    return message


def example_3_bulk_messages():
    """مثال 3: تخصيص رسائل متعددة"""
    print("\n" + "=" * 60)
    print("مثال 3: تخصيص رسائل لعدة أعمال")
    print("=" * 60)
    
    # جلب أول 5 أعمال من قاعدة البيانات
    db = SessionLocal()
    businesses = db.query(Business).limit(5).all()
    
    if not businesses:
        print("⚠️  لا توجد أعمال في قاعدة البيانات. شغّل example_1 أولاً")
        db.close()
        return
    
    businesses_data = [b.to_dict() for b in businesses]
    db.close()
    
    # قالب الرسالة
    template = """مرحباً {name} 👋

نحن شركة متخصصة في التسويق الرقمي ونود مساعدتك في تنمية أعمالك.

هل أنت مهتم بمعرفة المزيد؟ 🚀"""
    
    # تخصيص الرسائل
    ai = AIAssistant()
    messages = ai.customize_message_bulk(businesses_data, template)
    
    print(f"\n✅ تم تخصيص {len(messages)} رسالة")
    
    # عرض أول رسالتين
    for i, (business_id, message) in enumerate(list(messages.items())[:2], 1):
        business = next(b for b in businesses_data if b['id'] == business_id)
        print(f"\n{i}. رسالة إلى: {business['name']}")
        print("-" * 60)
        print(message)
    
    return messages


def example_4_send_whatsapp():
    """مثال 4: إرسال رسائل WhatsApp"""
    print("\n" + "=" * 60)
    print("مثال 4: إرسال رسائل WhatsApp")
    print("=" * 60)
    
    sender = WhatsAppSender()
    
    # إرسال رسالة واحدة (محاكاة)
    result = sender.send_message(
        phone_number="+966501234567",
        message="مرحباً! هذه رسالة تجريبية من نظام التسويق الاحترافي 🚀"
    )
    
    if result['success']:
        print(f"\n✅ تم إرسال الرسالة بنجاح")
        print(f"   الرقم: {result['phone']}")
    else:
        print(f"\n❌ فشل الإرسال: {result['error']}")
    
    return result


def example_5_complete_campaign():
    """مثال 5: حملة تسويقية كاملة"""
    print("\n" + "=" * 60)
    print("مثال 5: حملة تسويقية كاملة")
    print("=" * 60)
    
    # 1. البحث عن أعمال
    print("\n📍 الخطوة 1: البحث عن الأعمال...")
    scraper = GoogleMapsScraper()
    businesses = scraper.search_places("صيدليات", "الرياض")
    print(f"   ✅ تم العثور على {len(businesses)} صيدلية")
    
    # 2. تخصيص الرسائل
    print("\n📍 الخطوة 2: تخصيص الرسائل...")
    template = """مرحباً {name} 👋

نحن نقدم حلول تسويقية متخصصة للصيدليات.

✨ خدماتنا:
• إدارة التواصل الاجتماعي
• تصميم العروض والإعلانات
• نظام الطلب والتوصيل

هل تود معرفة المزيد؟"""
    
    ai = AIAssistant(use_deepseek=True)
    messages_dict = ai.customize_message_bulk(businesses, template)
    print(f"   ✅ تم تخصيص {len(messages_dict)} رسالة")
    
    # 3. تحضير رسائل WhatsApp
    print("\n📍 الخطوة 3: تحضير رسائل WhatsApp...")
    
    # ربط كل business بالرسالة المخصصة
    for business in businesses:
        business_id = business.get('id') or business.get('place_id')
        if business_id in messages_dict:
            business['custom_message'] = messages_dict[business_id]
    
    # تصفية الأعمال التي لديها أرقام
    businesses_with_phones = [b for b in businesses if b.get('phone')]
    print(f"   ✅ {len(businesses_with_phones)} عمل لديه رقم هاتف")
    
    # 4. إرسال الرسائل (محاكاة)
    print("\n📍 الخطوة 4: إرسال رسائل WhatsApp...")
    sender = WhatsAppSender()
    
    # تحضير قائمة الرسائل
    messages_to_send = []
    for business in businesses_with_phones[:3]:  # أول 3 فقط للتجربة
        messages_to_send.append({
            'phone': business['phone'],
            'message': business.get('custom_message', template.format(name=business['name'])),
            'business_id': business.get('id')
        })
    
    results = sender.send_bulk_messages(messages_to_send)
    
    # 5. النتائج
    print("\n" + "=" * 60)
    print("📊 نتائج الحملة:")
    print("=" * 60)
    print(f"إجمالي الأعمال المستخرجة: {len(businesses)}")
    print(f"الأعمال مع أرقام هواتف: {len(businesses_with_phones)}")
    print(f"الرسائل المرسلة بنجاح: {results['sent']}")
    print(f"الرسائل الفاشلة: {results['failed']}")
    print(f"معدل النجاح: {(results['sent']/results['total']*100):.1f}%")
    
    return results


def example_6_database_queries():
    """مثال 6: الاستعلام من قاعدة البيانات"""
    print("\n" + "=" * 60)
    print("مثال 6: استعلامات قاعدة البيانات")
    print("=" * 60)
    
    db = SessionLocal()
    
    # إحصائيات
    total_businesses = db.query(Business).count()
    businesses_with_phone = db.query(Business).filter(Business.phone != None).count()
    high_rated = db.query(Business).filter(Business.rating >= 4.0).count()
    
    print(f"\n📊 الإحصائيات:")
    print(f"   إجمالي الأعمال: {total_businesses}")
    print(f"   أعمال بأرقام: {businesses_with_phone}")
    print(f"   تقييم 4+ نجوم: {high_rated}")
    
    # أعلى 5 أعمال تقييماً
    top_businesses = db.query(Business).filter(
        Business.rating != None
    ).order_by(
        Business.rating.desc()
    ).limit(5).all()
    
    print(f"\n⭐ أعلى 5 أعمال تقييماً:")
    for i, business in enumerate(top_businesses, 1):
        print(f"   {i}. {business.name} - ⭐ {business.rating}")
    
    db.close()


def run_all_examples():
    """تشغيل جميع الأمثلة"""
    print("\n")
    print("╔═══════════════════════════════════════════════════════════╗")
    print("║                                                           ║")
    print("║         🚀 أمثلة استخدام النظام - Python 🚀              ║")
    print("║                                                           ║")
    print("╚═══════════════════════════════════════════════════════════╝")
    
    try:
        # example_1_search_businesses()
        # time.sleep(2)
        
        # example_2_generate_ai_messages()
        # time.sleep(2)
        
        # example_3_bulk_messages()
        # time.sleep(2)
        
        # example_4_send_whatsapp()
        # time.sleep(2)
        
        # example_5_complete_campaign()
        # time.sleep(2)
        
        example_6_database_queries()
        
        print("\n" + "=" * 60)
        print("✅ تم تنفيذ جميع الأمثلة بنجاح!")
        print("=" * 60)
        
    except Exception as e:
        print(f"\n❌ خطأ: {str(e)}")
        print("\n💡 تأكد من:")
        print("   1. وجود ملف .env بالمفاتيح الصحيحة")
        print("   2. تشغيل: python database.py")
        print("   3. وجود بيانات في قاعدة البيانات")


if __name__ == "__main__":
    # اختر مثال واحد أو شغّل الكل
    
    # مثال واحد:
    # example_1_search_businesses()
    # example_2_generate_ai_messages()
    # example_3_bulk_messages()
    # example_4_send_whatsapp()
    # example_5_complete_campaign()
    example_6_database_queries()
    
    # أو شغّل الكل:
    # run_all_examples()



