from datetime import datetime
from sqlalchemy import create_engine, Column, Integer, String, DateTime, Text, Float, Boolean
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import sessionmaker
from config import Config
import json

Base = declarative_base()
engine = create_engine(Config.DATABASE_URL)
SessionLocal = sessionmaker(bind=engine)

class Business(Base):
    """جدول لحفظ بيانات الأعمال المستخرجة"""
    __tablename__ = 'businesses'
    
    id = Column(Integer, primary_key=True)
    name = Column(String(255), nullable=False)
    place_id = Column(String(255), unique=True)
    address = Column(Text)
    phone = Column(String(50))
    website = Column(String(500))
    rating = Column(Float)
    total_ratings = Column(Integer)
    category = Column(String(255))
    location_lat = Column(Float)
    location_lng = Column(Float)
    search_query = Column(String(255))
    search_location = Column(String(255))
    created_at = Column(DateTime, default=datetime.utcnow)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'place_id': self.place_id,
            'address': self.address,
            'phone': self.phone,
            'website': self.website,
            'rating': self.rating,
            'total_ratings': self.total_ratings,
            'category': self.category,
            'location': {
                'lat': self.location_lat,
                'lng': self.location_lng
            } if self.location_lat and self.location_lng else None,
            'search_query': self.search_query,
            'search_location': self.search_location,
            'created_at': self.created_at.isoformat() if self.created_at else None
        }

class WhatsAppMessage(Base):
    """جدول لحفظ سجل رسائل الواتساب"""
    __tablename__ = 'whatsapp_messages'
    
    id = Column(Integer, primary_key=True)
    business_id = Column(Integer)
    phone_number = Column(String(50), nullable=False)
    message_content = Column(Text, nullable=False)
    status = Column(String(50))  # sent, delivered, failed, read
    sent_at = Column(DateTime, default=datetime.utcnow)
    delivered_at = Column(DateTime)
    error_message = Column(Text)
    
    def to_dict(self):
        return {
            'id': self.id,
            'business_id': self.business_id,
            'phone_number': self.phone_number,
            'message_content': self.message_content,
            'status': self.status,
            'sent_at': self.sent_at.isoformat() if self.sent_at else None,
            'delivered_at': self.delivered_at.isoformat() if self.delivered_at else None,
            'error_message': self.error_message
        }

class Campaign(Base):
    """جدول لحفظ الحملات التسويقية"""
    __tablename__ = 'campaigns'
    
    id = Column(Integer, primary_key=True)
    name = Column(String(255), nullable=False)
    description = Column(Text)
    search_query = Column(String(255))
    search_location = Column(String(255))
    message_template = Column(Text)
    total_businesses = Column(Integer, default=0)
    messages_sent = Column(Integer, default=0)
    messages_delivered = Column(Integer, default=0)
    messages_failed = Column(Integer, default=0)
    is_active = Column(Boolean, default=True)
    created_at = Column(DateTime, default=datetime.utcnow)
    completed_at = Column(DateTime)
    
    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'description': self.description,
            'search_query': self.search_query,
            'search_location': self.search_location,
            'message_template': self.message_template,
            'total_businesses': self.total_businesses,
            'messages_sent': self.messages_sent,
            'messages_delivered': self.messages_delivered,
            'messages_failed': self.messages_failed,
            'is_active': self.is_active,
            'created_at': self.created_at.isoformat() if self.created_at else None,
            'completed_at': self.completed_at.isoformat() if self.completed_at else None
        }

def init_db():
    """إنشاء جميع الجداول"""
    Base.metadata.create_all(engine)
    print("✅ تم إنشاء قاعدة البيانات بنجاح")

def get_db():
    """الحصول على جلسة قاعدة البيانات"""
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()

if __name__ == "__main__":
    init_db()



