"""
نسخ احتياطي واستعادة قاعدة البيانات
"""

import shutil
import os
from datetime import datetime
import json
from database import SessionLocal, Business, WhatsAppMessage, Campaign

DATABASE_FILE = 'marketing_system.db'
BACKUP_DIR = 'backups'

def create_backup():
    """إنشاء نسخة احتياطية من قاعدة البيانات"""
    print("\n📦 إنشاء نسخة احتياطية...")
    
    # التحقق من وجود قاعدة البيانات
    if not os.path.exists(DATABASE_FILE):
        print("❌ قاعدة البيانات غير موجودة!")
        return None
    
    # إنشاء مجلد النسخ الاحتياطية
    if not os.path.exists(BACKUP_DIR):
        os.makedirs(BACKUP_DIR)
        print(f"✅ تم إنشاء مجلد {BACKUP_DIR}")
    
    # اسم ملف النسخة الاحتياطية
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    backup_file = os.path.join(BACKUP_DIR, f'backup_{timestamp}.db')
    
    # نسخ قاعدة البيانات
    try:
        shutil.copy2(DATABASE_FILE, backup_file)
        file_size = os.path.getsize(backup_file) / 1024  # KB
        
        print(f"✅ تم إنشاء النسخة الاحتياطية بنجاح")
        print(f"📁 الملف: {backup_file}")
        print(f"📊 الحجم: {file_size:.2f} KB")
        
        # حفظ معلومات النسخة
        save_backup_info(backup_file, file_size)
        
        return backup_file
        
    except Exception as e:
        print(f"❌ خطأ في إنشاء النسخة الاحتياطية: {str(e)}")
        return None


def save_backup_info(backup_file, file_size):
    """حفظ معلومات النسخة الاحتياطية"""
    db = SessionLocal()
    
    info = {
        'backup_file': backup_file,
        'timestamp': datetime.now().isoformat(),
        'file_size_kb': file_size,
        'total_businesses': db.query(Business).count(),
        'total_messages': db.query(WhatsAppMessage).count(),
        'total_campaigns': db.query(Campaign).count()
    }
    
    db.close()
    
    # حفظ في ملف JSON
    info_file = backup_file.replace('.db', '_info.json')
    with open(info_file, 'w', encoding='utf-8') as f:
        json.dump(info, f, ensure_ascii=False, indent=2)


def list_backups():
    """عرض قائمة النسخ الاحتياطية"""
    print("\n📂 النسخ الاحتياطية المتوفرة:\n")
    
    if not os.path.exists(BACKUP_DIR):
        print("⚠️  لا توجد نسخ احتياطية")
        return []
    
    backups = []
    
    for filename in os.listdir(BACKUP_DIR):
        if filename.endswith('.db'):
            filepath = os.path.join(BACKUP_DIR, filename)
            file_size = os.path.getsize(filepath) / 1024  # KB
            file_time = datetime.fromtimestamp(os.path.getmtime(filepath))
            
            backups.append({
                'filename': filename,
                'filepath': filepath,
                'size': file_size,
                'time': file_time
            })
    
    # ترتيب حسب التاريخ
    backups.sort(key=lambda x: x['time'], reverse=True)
    
    if not backups:
        print("⚠️  لا توجد نسخ احتياطية")
        return []
    
    # عرض القائمة
    for i, backup in enumerate(backups, 1):
        print(f"{i}. {backup['filename']}")
        print(f"   📅 {backup['time'].strftime('%Y-%m-%d %H:%M:%S')}")
        print(f"   📊 {backup['size']:.2f} KB")
        
        # محاولة قراءة معلومات إضافية
        info_file = backup['filepath'].replace('.db', '_info.json')
        if os.path.exists(info_file):
            try:
                with open(info_file, 'r', encoding='utf-8') as f:
                    info = json.load(f)
                    print(f"   📊 أعمال: {info.get('total_businesses', 0)}")
                    print(f"   📊 رسائل: {info.get('total_messages', 0)}")
                    print(f"   📊 حملات: {info.get('total_campaigns', 0)}")
            except:
                pass
        
        print()
    
    return backups


def restore_backup(backup_file):
    """استعادة نسخة احتياطية"""
    print(f"\n♻️  استعادة النسخة الاحتياطية: {backup_file}\n")
    
    if not os.path.exists(backup_file):
        print(f"❌ الملف غير موجود: {backup_file}")
        return False
    
    try:
        # إنشاء نسخة احتياطية من الحالة الحالية أولاً
        if os.path.exists(DATABASE_FILE):
            print("📦 إنشاء نسخة احتياطية من الحالة الحالية...")
            current_backup = DATABASE_FILE + '.before_restore'
            shutil.copy2(DATABASE_FILE, current_backup)
            print(f"✅ تم حفظ النسخة الحالية في: {current_backup}")
        
        # استعادة النسخة الاحتياطية
        print(f"♻️  جارٍ الاستعادة...")
        shutil.copy2(backup_file, DATABASE_FILE)
        
        print(f"✅ تم استعادة النسخة الاحتياطية بنجاح!")
        return True
        
    except Exception as e:
        print(f"❌ خطأ في الاستعادة: {str(e)}")
        return False


def auto_backup():
    """نسخ احتياطي تلقائي مع حذف النسخ القديمة"""
    print("\n🔄 نسخ احتياطي تلقائي...")
    
    # إنشاء نسخة جديدة
    backup_file = create_backup()
    
    if not backup_file:
        return
    
    # حذف النسخ القديمة (الاحتفاظ بآخر 10)
    backups = list_backups()
    
    if len(backups) > 10:
        print(f"\n🗑️  حذف النسخ القديمة (الاحتفاظ بآخر 10)...")
        for backup in backups[10:]:
            try:
                os.remove(backup['filepath'])
                info_file = backup['filepath'].replace('.db', '_info.json')
                if os.path.exists(info_file):
                    os.remove(info_file)
                print(f"   🗑️  تم حذف: {backup['filename']}")
            except Exception as e:
                print(f"   ❌ خطأ في حذف {backup['filename']}: {str(e)}")


def interactive_restore():
    """استعادة تفاعلية"""
    backups = list_backups()
    
    if not backups:
        return
    
    print("=" * 60)
    print("اختر رقم النسخة للاستعادة (0 للإلغاء):")
    
    try:
        choice = int(input("الخيار: "))
        
        if choice == 0:
            print("تم الإلغاء")
            return
        
        if 1 <= choice <= len(backups):
            backup = backups[choice - 1]
            
            confirm = input(f"\n⚠️  هل أنت متأكد من استعادة {backup['filename']}؟ (y/n): ")
            
            if confirm.lower() == 'y':
                restore_backup(backup['filepath'])
            else:
                print("تم الإلغاء")
        else:
            print("❌ خيار غير صحيح")
            
    except ValueError:
        print("❌ يرجى إدخال رقم صحيح")
    except KeyboardInterrupt:
        print("\n\nتم الإلغاء")


def main():
    """القائمة الرئيسية"""
    print("\n╔═══════════════════════════════════════════════════════════╗")
    print("║                                                           ║")
    print("║         💾 نظام النسخ الاحتياطي واستعادة 💾              ║")
    print("║                                                           ║")
    print("╚═══════════════════════════════════════════════════════════╝")
    
    while True:
        print("\n" + "=" * 60)
        print("الخيارات:")
        print("=" * 60)
        print("1. إنشاء نسخة احتياطية")
        print("2. عرض النسخ الاحتياطية")
        print("3. استعادة نسخة احتياطية")
        print("4. نسخ احتياطي تلقائي (مع حذف القديمة)")
        print("0. خروج")
        print("=" * 60)
        
        try:
            choice = input("\nاختر (0-4): ")
            
            if choice == '1':
                create_backup()
            elif choice == '2':
                list_backups()
            elif choice == '3':
                interactive_restore()
            elif choice == '4':
                auto_backup()
            elif choice == '0':
                print("\n👋 إلى اللقاء!")
                break
            else:
                print("❌ خيار غير صحيح")
                
        except KeyboardInterrupt:
            print("\n\n👋 إلى اللقاء!")
            break


if __name__ == "__main__":
    # استخدام مباشر
    # create_backup()
    # list_backups()
    
    # أو القائمة التفاعلية
    main()



