from flask import Flask, request, jsonify, render_template, send_from_directory
from flask_cors import CORS
from config import Config
from database import init_db, SessionLocal, Business, Campaign, WhatsAppMessage
from google_maps_scraper import GoogleMapsScraper
from ai_integration import AIAssistant
from whatsapp_integration import WhatsAppSender, prepare_bulk_messages
import logging
from datetime import datetime
import os

# إعداد السجلات
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# إنشاء التطبيق
app = Flask(__name__, static_folder='static', template_folder='templates')
app.config.from_object(Config)
CORS(app)

# التحقق من المفاتيح وإنشاء قاعدة البيانات
try:
    Config.validate()
    init_db()
    logger.info("✅ تم تهيئة التطبيق بنجاح")
except Exception as e:
    logger.error(f"❌ خطأ في التهيئة: {str(e)}")

# الصفحة الرئيسية
@app.route('/')
def index():
    return render_template('index.html')

# ========================
# Google Maps APIs
# ========================

@app.route('/api/search', methods=['POST'])
def search_places():
    """
    البحث عن الأماكن في خرائط جوجل
    
    Body:
        query: نوع العمل (مثال: مطاعم، صيدليات)
        location: المنطقة أو المدينة
    """
    try:
        data = request.json
        query = data.get('query')
        location = data.get('location')
        
        if not query or not location:
            return jsonify({
                'success': False,
                'error': 'يجب إدخال نوع العمل والموقع'
            }), 400
        
        logger.info(f"🔍 بحث جديد: {query} في {location}")
        
        scraper = GoogleMapsScraper()
        businesses = scraper.search_places(query, location)
        
        return jsonify({
            'success': True,
            'count': len(businesses),
            'data': businesses
        })
        
    except Exception as e:
        logger.error(f"❌ خطأ في البحث: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/api/businesses', methods=['GET'])
def get_businesses():
    """الحصول على جميع الأعمال المحفوظة"""
    try:
        db = SessionLocal()
        limit = request.args.get('limit', 100, type=int)
        search_query = request.args.get('search', '')
        
        query = db.query(Business)
        
        if search_query:
            query = query.filter(
                (Business.name.contains(search_query)) |
                (Business.category.contains(search_query)) |
                (Business.search_location.contains(search_query))
            )
        
        businesses = query.limit(limit).all()
        
        return jsonify({
            'success': True,
            'count': len(businesses),
            'data': [b.to_dict() for b in businesses]
        })
        
    except Exception as e:
        logger.error(f"❌ خطأ في جلب البيانات: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500
    finally:
        db.close()

# ========================
# AI APIs
# ========================

@app.route('/api/ai/generate-message', methods=['POST'])
def generate_message():
    """
    توليد رسالة تسويقية بالذكاء الاصطناعي
    
    Body:
        business_data: بيانات العمل
        custom_prompt: نص إضافي (اختياري)
        use_deepseek: استخدام DeepSeek بدلاً من ChatGPT (اختياري)
    """
    try:
        data = request.json
        business_data = data.get('business_data')
        custom_prompt = data.get('custom_prompt')
        use_deepseek = data.get('use_deepseek', False)
        
        if not business_data:
            return jsonify({
                'success': False,
                'error': 'بيانات العمل مطلوبة'
            }), 400
        
        ai = AIAssistant(use_deepseek=use_deepseek)
        message = ai.generate_marketing_message(business_data, custom_prompt)
        
        return jsonify({
            'success': True,
            'message': message
        })
        
    except Exception as e:
        logger.error(f"❌ خطأ في توليد الرسالة: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/api/ai/customize-bulk', methods=['POST'])
def customize_bulk_messages():
    """
    تخصيص رسالة لعدة أعمال
    
    Body:
        business_ids: قائمة معرفات الأعمال
        template: قالب الرسالة
    """
    try:
        data = request.json
        business_ids = data.get('business_ids', [])
        template = data.get('template')
        
        if not business_ids or not template:
            return jsonify({
                'success': False,
                'error': 'معرفات الأعمال وقالب الرسالة مطلوبة'
            }), 400
        
        db = SessionLocal()
        businesses = db.query(Business).filter(Business.id.in_(business_ids)).all()
        businesses_data = [b.to_dict() for b in businesses]
        
        ai = AIAssistant()
        messages = ai.customize_message_bulk(businesses_data, template)
        
        return jsonify({
            'success': True,
            'messages': messages
        })
        
    except Exception as e:
        logger.error(f"❌ خطأ في تخصيص الرسائل: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500
    finally:
        db.close()

# ========================
# WhatsApp APIs
# ========================

@app.route('/api/whatsapp/send', methods=['POST'])
def send_whatsapp():
    """
    إرسال رسالة واتساب واحدة
    
    Body:
        phone: رقم الهاتف
        message: نص الرسالة
        business_id: معرف العمل (اختياري)
    """
    try:
        data = request.json
        phone = data.get('phone')
        message = data.get('message')
        business_id = data.get('business_id')
        
        if not phone or not message:
            return jsonify({
                'success': False,
                'error': 'رقم الهاتف والرسالة مطلوبة'
            }), 400
        
        sender = WhatsAppSender()
        result = sender.send_message(phone, message, business_id)
        
        return jsonify(result)
        
    except Exception as e:
        logger.error(f"❌ خطأ في إرسال الرسالة: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500

@app.route('/api/whatsapp/send-bulk', methods=['POST'])
def send_bulk_whatsapp():
    """
    إرسال رسائل متعددة
    
    Body:
        business_ids: قائمة معرفات الأعمال
        template: قالب الرسالة
    """
    try:
        data = request.json
        business_ids = data.get('business_ids', [])
        template = data.get('template')
        
        if not business_ids or not template:
            return jsonify({
                'success': False,
                'error': 'معرفات الأعمال وقالب الرسالة مطلوبة'
            }), 400
        
        db = SessionLocal()
        businesses = db.query(Business).filter(Business.id.in_(business_ids)).all()
        businesses_data = [b.to_dict() for b in businesses]
        
        # تحضير الرسائل
        messages = prepare_bulk_messages(businesses_data, template)
        
        # إرسال الرسائل
        sender = WhatsAppSender()
        results = sender.send_bulk_messages(messages)
        
        return jsonify({
            'success': True,
            'results': results
        })
        
    except Exception as e:
        logger.error(f"❌ خطأ في إرسال الرسائل: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500
    finally:
        db.close()

# ========================
# Campaign APIs
# ========================

@app.route('/api/campaigns', methods=['GET'])
def get_campaigns():
    """الحصول على جميع الحملات"""
    try:
        db = SessionLocal()
        campaigns = db.query(Campaign).order_by(Campaign.created_at.desc()).all()
        
        return jsonify({
            'success': True,
            'count': len(campaigns),
            'data': [c.to_dict() for c in campaigns]
        })
        
    except Exception as e:
        logger.error(f"❌ خطأ في جلب الحملات: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500
    finally:
        db.close()

@app.route('/api/campaigns', methods=['POST'])
def create_campaign():
    """إنشاء حملة جديدة"""
    try:
        data = request.json
        
        db = SessionLocal()
        campaign = Campaign(
            name=data.get('name'),
            description=data.get('description'),
            search_query=data.get('search_query'),
            search_location=data.get('search_location'),
            message_template=data.get('message_template')
        )
        
        db.add(campaign)
        db.commit()
        db.refresh(campaign)
        
        return jsonify({
            'success': True,
            'campaign': campaign.to_dict()
        })
        
    except Exception as e:
        logger.error(f"❌ خطأ في إنشاء الحملة: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500
    finally:
        db.close()

# ========================
# Statistics APIs
# ========================

@app.route('/api/stats', methods=['GET'])
def get_statistics():
    """الحصول على إحصائيات النظام"""
    try:
        db = SessionLocal()
        
        total_businesses = db.query(Business).count()
        total_campaigns = db.query(Campaign).count()
        total_messages = db.query(WhatsAppMessage).count()
        sent_messages = db.query(WhatsAppMessage).filter_by(status='sent').count()
        failed_messages = db.query(WhatsAppMessage).filter_by(status='failed').count()
        
        return jsonify({
            'success': True,
            'stats': {
                'total_businesses': total_businesses,
                'total_campaigns': total_campaigns,
                'total_messages': total_messages,
                'sent_messages': sent_messages,
                'failed_messages': failed_messages,
                'success_rate': round((sent_messages / total_messages * 100) if total_messages > 0 else 0, 2)
            }
        })
        
    except Exception as e:
        logger.error(f"❌ خطأ في جلب الإحصائيات: {str(e)}")
        return jsonify({
            'success': False,
            'error': str(e)
        }), 500
    finally:
        db.close()

if __name__ == '__main__':
    print("""
    ╔═══════════════════════════════════════════════════════════╗
    ║                                                           ║
    ║        🚀 نظام التسويق الاحترافي المتقدم 🚀              ║
    ║                                                           ║
    ║  ✅ استخراج البيانات من Google Maps                     ║
    ║  ✅ ذكاء اصطناعي متقدم                                  ║
    ║  ✅ إرسال رسائل WhatsApp                                ║
    ║  ✅ إدارة الحملات التسويقية                            ║
    ║                                                           ║
    ╚═══════════════════════════════════════════════════════════╝
    
    🌐 التطبيق يعمل على: http://localhost:5000
    📚 الوثائق: http://localhost:5000/api/docs
    
    """)
    
    app.run(debug=True, host='0.0.0.0', port=5000)



