import openai
import requests
from typing import Dict, Optional
from config import Config
import logging

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class AIAssistant:
    """نظام الذكاء الاصطناعي للردود الذكية وتخصيص الرسائل"""
    
    def __init__(self, use_deepseek: bool = False):
        self.use_deepseek = use_deepseek
        
        if use_deepseek:
            self.api_key = Config.DEEPSEEK_API_KEY
            self.api_base = "https://api.deepseek.com/v1"
        else:
            openai.api_key = Config.OPENAI_API_KEY
    
    def generate_marketing_message(self, business_data: Dict, custom_prompt: Optional[str] = None) -> str:
        """
        توليد رسالة تسويقية مخصصة باستخدام الذكاء الاصطناعي
        
        Args:
            business_data: بيانات العمل التجاري
            custom_prompt: نص إضافي لتخصيص الرسالة
            
        Returns:
            رسالة تسويقية مخصصة
        """
        try:
            business_name = business_data.get('name', 'عزيزي العميل')
            category = business_data.get('category', 'العمل')
            
            system_prompt = """أنت خبير في التسويق والمبيعات. مهمتك إنشاء رسائل تسويقية احترافية ومؤثرة باللغة العربية.
الرسالة يجب أن تكون:
- قصيرة ومباشرة (لا تتجاوز 3-4 أسطر)
- مهنية ومحترمة
- تجذب الانتباه
- تحتوي على دعوة واضحة للتواصل
- مناسبة لإرسالها عبر الواتساب"""
            
            user_prompt = f"""اكتب رسالة تسويقية لـ:
اسم العمل: {business_name}
نوع العمل: {category}

{"معلومات إضافية: " + custom_prompt if custom_prompt else ""}

الرسالة يجب أن تعرض خدماتنا بطريقة جذابة وتشجع على التواصل."""

            if self.use_deepseek:
                message = self._call_deepseek(system_prompt, user_prompt)
            else:
                message = self._call_openai(system_prompt, user_prompt)
            
            return message
            
        except Exception as e:
            logger.error(f"❌ خطأ في توليد الرسالة: {str(e)}")
            # رسالة افتراضية في حالة الفشل
            return f"""مرحباً {business_name} 👋

نحن نقدم حلول تسويقية متقدمة لمساعدة أعمالك على النمو والوصول لعملاء جدد.

هل أنت مهتم بمعرفة المزيد عن خدماتنا؟

للتواصل: [رقم الواتساب]"""
    
    def _call_openai(self, system_prompt: str, user_prompt: str) -> str:
        """استدعاء OpenAI API"""
        try:
            response = openai.chat.completions.create(
                model="gpt-4",
                messages=[
                    {"role": "system", "content": system_prompt},
                    {"role": "user", "content": user_prompt}
                ],
                temperature=Config.AI_TEMPERATURE,
                max_tokens=Config.AI_MAX_TOKENS
            )
            
            return response.choices[0].message.content.strip()
            
        except Exception as e:
            logger.error(f"❌ خطأ في OpenAI API: {str(e)}")
            raise
    
    def _call_deepseek(self, system_prompt: str, user_prompt: str) -> str:
        """استدعاء DeepSeek API"""
        try:
            headers = {
                "Authorization": f"Bearer {self.api_key}",
                "Content-Type": "application/json"
            }
            
            data = {
                "model": "deepseek-chat",
                "messages": [
                    {"role": "system", "content": system_prompt},
                    {"role": "user", "content": user_prompt}
                ],
                "temperature": Config.AI_TEMPERATURE,
                "max_tokens": Config.AI_MAX_TOKENS
            }
            
            response = requests.post(
                f"{self.api_base}/chat/completions",
                headers=headers,
                json=data,
                timeout=Config.REQUEST_TIMEOUT
            )
            
            response.raise_for_status()
            result = response.json()
            
            return result['choices'][0]['message']['content'].strip()
            
        except Exception as e:
            logger.error(f"❌ خطأ في DeepSeek API: {str(e)}")
            raise
    
    def customize_message_bulk(self, businesses: list, template: str) -> Dict[str, str]:
        """
        تخصيص رسالة واحدة لعدة أعمال تجارية
        
        Args:
            businesses: قائمة الأعمال التجارية
            template: قالب الرسالة مع متغيرات مثل {name}, {category}
            
        Returns:
            قاموس يربط business_id بالرسالة المخصصة
        """
        messages = {}
        
        for business in businesses:
            try:
                # استبدال المتغيرات في القالب
                message = template.format(
                    name=business.get('name', 'عزيزي العميل'),
                    category=business.get('category', ''),
                    address=business.get('address', ''),
                    rating=business.get('rating', ''),
                )
                
                messages[business.get('id') or business.get('place_id')] = message
                
            except Exception as e:
                logger.error(f"❌ خطأ في تخصيص الرسالة لـ {business.get('name')}: {str(e)}")
                continue
        
        return messages
    
    def generate_campaign_name(self, query: str, location: str) -> str:
        """توليد اسم احترافي للحملة"""
        try:
            prompt = f"اقترح اسماً قصيراً وجذاباً لحملة تسويقية تستهدف '{query}' في '{location}'. الاسم فقط بدون شرح."
            
            if self.use_deepseek:
                return self._call_deepseek("أنت مساعد تسويق", prompt)
            else:
                return self._call_openai("أنت مساعد تسويق", prompt)
                
        except:
            return f"حملة {query} - {location}"



